module net.BurtonRadons.dedit.view;

import net.BurtonRadons.dedit.main;

class View : Canvas
{
    Document document;

    int pageHeight; /* Number of lines in a page. */

    int baseX = 4;

    bit dirty = true;
    int caretTime;

    typedef void delegate (View v) Command;
    alias char [] charArray;
    static Command [char []] commands;
    static charArray [char []] commandDescs;
    static charArray [char []] commandOptions;
    static charArray [char []] commandHelp;
    charArray [char []] bindings; /* No-shift, no-control bindings. */
    charArray [char []] controlBindings; /* No-shift, control bindings. */
    charArray [char []] shiftBindings; /* Shift, no-control bindings. */
    charArray [char []] shiftControlBindings; /* Shift, control bindings. */

    bit runRecurse = false;
    bit paintDefer = false;
    bit paintCalled = false;
    bit typing; /**< If set, the user is typing, so runChar shouldn't create a mark. */

    char [] projectName; /* Project file this comes from. */
    Document [] documents; /* Documents sorted in filename order. */
    Document [] ordered; /* Documents sorted in order of use. */
    MenuCommand menu;

    ProjectView.Folder fCurrentFolder; /* Current folder or null to use document.folder. */
    
    char [] [] macro; /**< Recorded macro, a list of commands to execute.  Characters are encoded by "*c", where c is the character to type. */
    char [] [] macroNext; /**< The currently recording macro, which is put into macro when recording is off.  This way allows combining the previous macro and the next one. */
    bit macroRecording; /**< Whether the macro is currently recording. */
    
    ExternalTool compileCommand; /**< Command to execute for compilation (ProjectCompile). */
    ExternalTool runCommand; /**< Command to execute for run (ProjectRun). */

    bit delegate (View view, Document document, inout int line, inout int offset, out int endLine, out int endOffset) find;

    bit findDefault (View view, Document document, inout int line, inout int offset, out int endLine, out int endOffset) { return false; }

    void doChar (Event e)
    {
        runChar (e.keyCode [0]);
    }

    this (Control parent)
    {
        compileCommand = new ExternalTool;
        runCommand = new ExternalTool;
        find = &findDefault;

        super (parent);
        doubleBuffer (true);

        onChar.add (&doChar);
        onKeyDown.add (&doKeyDown);
        onPaint.add (&doPaint);

        timer (500, &repaintCaret);

        registry.loadString (register ~ "project", projectName);

        onMouseMove.add (&doMouseMove);
        onLButtonDown.add (&doLButtonDown);
        onLButtonUp.add (&doLButtonUp);
        onRButtonDown.add (&doMakeFocus);
        onMouseWheel.add (&doMouseWheel); /* The parent window gives us this. */
        onRButtonUp.add (&doContextMenu);
        onVScroll.add (&doVScroll);
        onHScroll.add (&doHScroll);

        // name, keycode, shift, control
        bind ("LineDown", "Down");
        bind ("LineDownExtend", "Down", true, false);
        bind ("CharLeft", "Left");
        bind ("CharLeftExtend", "Left", true, false);
        bind ("SegmentLeft", "Left", false, true);
        bind ("SegmentLeftExtend", "Left", true, true);
        bind ("SegmentRight", "Right", false, true);
        bind ("SegmentRightExtend", "Right", true, true);
        bind ("CharRight", "Right");
        bind ("CharRightExtend", "Right", true, false);
        bind ("LineUp", "Up");
        bind ("LineUpExtend", "Up", true, false);
        bind ("HomeOrLineStart", "Home");
        bind ("HomeOrLineStartExtend", "Home", true, false);
        bind ("End", "End");
        bind ("EndExtend", "End", true, false);
        bind ("PageUp", "PageUp");
        bind ("PageUpExtend", "PageUp", true, false);
        bind ("PageDown", "PageDown");
        bind ("PageDownExtend", "PageDown", true, false);
        bind ("DeleteSelectionOrPrev", "BackSpace");
        bind ("DeleteSelectionOrNext", "Delete");
        bind ("Return", "Return");
        bind ("Return", "Return", true, false);
        bind ("DeleteLine", "Y", false, true);
        bind ("FileSave", "S", false, true);
        bind ("DocumentStart", "Home", false, true);
        bind ("DocumentStartExtend", "Home", true, true);
        bind ("DocumentEnd", "End", false, true);
        bind ("DocumentEndExtend", "End", true, true);
        bind ("SelectAll", "A", false, true);
        bind ("Copy", "C", false, true);
        bind ("Cut", "X", false, true);
        bind ("Paste", "V", false, true);
        bind ("Cut", "Delete", true, false);
        bind ("Paste", "Insert", true, false);
        bind ("Undo", "Z", false, true);
        bind ("Redo", "Y", false, true);
        bind ("Indent", "Tab", false, false);
        bind ("Dedent", "Tab", true, false);
        bind ("Find", "F", false, true);
        bind ("FindNext", "F3", false, false);
        bind ("ScrollUp", "Up", false, true);
        bind ("ScrollDown", "Down", false, true);
        bind ("MacroRecordToggle", "R", true, true);
        bind ("MacroPlay", "P", true, true);
        bind ("ProjectCompile", "F5", false, false);
        bind ("ProjectRun", "F7", false, false);

        menu = new MenuCommand ();
        menu.add ("Keyboard Map", "KeyboardMap");
        menu.add ("Language", "SelectSyntaxHighlighter");
        menu.add ("Options", "Options");
        
        cursor (Cursor.Text);
    }

    /* Set the current folder. */
    void currentFolder (ProjectView.Folder value)
    {
        fCurrentFolder = value;
    }

    /* Get the current folder. */
    ProjectView.Folder currentFolder ()
    {
        if (fCurrentFolder !== null)
            return fCurrentFolder;
        if (document)
            return global.projectView.findDocumentFolder (document);
        return global.projectView.root;
    }

    /* Ask for a confirmation dialog if buffers have been modified, and save them. */
    bit confirmModified ()
    {
        char [] [] names;

        for (int c; c < documents.length; c ++)
            if (documents [c].modified)
                names ~= documents [c].name ();

        if (names.length)
        {
            char [] text;

            if (names.length == 1)
                text = names [0] ~ " has been modified; save changes?";
            else
            {
                text = "These files have been modified; save changes?\n\n";
                for (int c; c < names.length; c ++)
                {
                    text ~= names [c];
                    text ~= "\n";
                }
            }

            char [] r;

            r = messageBox ("Confirm Save", text, MB.YesNoCancel);

            if (r == "Yes")
            {
                for (int c; c < documents.length; c ++)
                    if (documents [c].modified && documents [c].filename !== null)
                        documents [c].save ();
                return true;
            }
            else if (r == "No")
                return true;
            return false;
        }

        return true;
    }

    /* Get the highlight color of a syntax coloring index. */
    Color highColor (char code)
    {
        switch (code)
        {
            case '*': return global.colorComment;
            case '#': return global.colorNumber;
            case '"': return global.colorString;
            case 'i': return global.colorIdentifier;
            case 'm': return global.colorSpecialIdentifier;
            case 'r': return global.colorKeyword;
            case 's': return global.colorSymbol;
            default:  return global.colorText;
        }
    }

    void clean ()
    {
        if (document.dirty || dirty)
        {
            document.dirty = false;
            dirty = false;
            paint ();
        }
    }

    /* Get the character offset of a point. */
    void findOffset (int line, int offset, out int x, out int y)
    {
        if (line >= document.lines.length)
        {
            x = y = 0;
            return;
        }
        
        char [] text = document.lines [line];

        document.cleanLine (line);
        x = getOffsetX (line, offset);//textWidth (text [0 .. offset]) + baseX;
        y = (line - document.lineOffset) * global.fontHeight;
    }

    /* Get the X offset of a point. */
    int getOffsetX (int line, int offset)
    {
        if (document.lines.length == 0) return 0;
        if (line < 0) line = 0;
        if (line >= document.lines.length) line = document.lines.length - 1;
        if (offset < 0) offset = 0;
        if (offset > document.lines [line].length) offset = document.lines [line].length;
        document.cleanLine (line);

        char [] text = document.lines [line] [0 .. offset];
        int c, s, x = baseX;
        int spaceWidth = textWidth (" ") * 1.5;
        TabParams tp = document.tabParams ();

        for ( ; c < text.length; c ++)
        {
            if (std.ctype.isspace (text [c]))
            {
                x += textWidth (text [s .. c]);
                s = c + 1;

                if (text [c] == ' ')
                    x += spaceWidth;
                else if (text [c] == '\t')
                    x += spaceWidth * tp.tabSize;
            }
        }

        x += textWidth (text [s .. c]);

        return x;
    }
    
    int getOffsetY (int line)
    {
        return (line - document.lineOffset) * global.fontHeight;
    }

    /* Get the nearest offset index to this x position. */
    int findNearestX (int line, int x)
    {
        char [] text = document.lines [line];
        int n, p, s, spaceWidth;
        TabParams tp = document.tabParams ();

        if (text.length == 0)
            return 0;
        x -= baseX - document.hscroll;
        font (global.font);
        spaceWidth = textWidth (" ") * 1.5;
        document.cleanLine (line);
        for (int c = 0; c < text.length; c ++, p = n)
        {
            if (text [c] == ' ')
            {
                x -= textWidth (text [s .. c]);
                s = c + 1;
                if (x < 0)
                    return imax (0, c - 1);
                x -= spaceWidth;
                if (x < 0)
                    return imax (0, c);
            }
            else if (text [c] == '\t')
            {
                x -= textWidth (text [s .. c]);
                s = c + 1;
                x -= spaceWidth * tp.tabSize;
                if (x < 0)
                    return imax (0, c - 1);
            }
            else
            {
                n = textWidth (text [s .. c]);
                if (n > x)
                    return imax (0, c - 1);
            }
        }

        if (n >= x)
            return text.length - 1;

        return text.length;
    }

    /* Get the line for this y position. */
    int findNearestY (int y)
    {
        int value = document.lineOffset + y / global.fontHeight;

        value = imax (0, value);
        value = imin (document.lines.length - 1, value);
        return value;
    }

    int caretX;

    /* Mark the caret offset and scroll to make the caret in the window. */
    void caretChange ()
    {
        caretX = getOffsetX (document.line, document.offset);

        caretInWindow ();
    }
    
    int visibleLineOffset;
    int visibleLine;
    int visibleOffset;
    int visibleSelStartLine = -1;
    int visibleSelStartOffset;
    int visibleSelEndLine;
    int visibleSelEndOffset;

    /* Scroll the window until the caret is in it. */
    void caretInWindow ()
    {
        int oldHScroll = document.hscroll;
        int oldVScroll = document.lineOffset;
        int oldCaretX = caretX;
        
        if (caretX < document.hscroll + visualWidth () * 0.1)
            document.hscroll = imax (0, caretX - visualWidth () * 0.1);
        else if (caretX > document.hscroll + visualWidth () * 0.9)
            document.hscroll = caretX - visualWidth () * 0.9;

        if (document.line > document.lineOffset + height () / global.fontHeight - 3)
            document.lineOffset = document.line - height () / global.fontHeight + 3;
        if (document.line < document.lineOffset + 3)
            document.lineOffset = document.line - 3;
        vscrollPoint (document.lineOffset);
        caretTime = elapsedTime ();
        dirty = true;
        
        if (oldHScroll != document.hscroll || oldVScroll != document.lineOffset || document.lineOffset != visibleLineOffset)
            paint ();
        else if (visibleLine != document.line || visibleOffset != document.offset)
        {
            dirtyCaret (visibleLine, visibleOffset);
            dirtyCaret (document.line, document.offset);
        }
        
        if (document.selStartLine != -1 && visibleSelStartLine != -1)
        {
            dirtySelection (visibleSelStartLine, visibleSelStartOffset, document.selStartLine, document.selStartOffset);
            dirtySelection (visibleSelEndLine, visibleSelEndOffset, document.selEndLine, document.selEndOffset);
        }
        else if (document.selStartLine != -1)
            dirtySelection (document.selStartLine, document.selStartOffset, document.selEndLine, document.selEndOffset);
        else
            dirtySelection (visibleSelStartLine, visibleSelStartOffset, visibleSelEndLine, visibleSelEndOffset);
    }
    
    void dirtySelection (int startLine, int startOffset, int endLine, int endOffset)
    {
        void swap (inout int a, inout int b)
        {
            int c = a;
            
            a = b;
            b = c;
        }
        
        if (startLine == -1 || endLine == -1)
            return;
       
        if (startLine >= document.lines.length)
        {
            startLine = document.lines.length - 1;
            startOffset = 0;
        }
        
        if (endLine >= document.lines.length)
        {
            endLine = document.lines.length - 1;
            endOffset = document.lines [endLine].length;
        }
        
        startOffset = imin (document.lines [startLine].length, startOffset);
        endOffset = imin (document.lines [endLine].length, endOffset);
        
        if (startLine > endLine || (startLine == endLine && startOffset > endOffset))
        {
            swap (startLine, endLine);
            swap (startOffset, endOffset);
        }
        
        if (startLine == endLine && startOffset == endOffset)
            return;
        
        if (startLine != endLine)
        {
            paintRegion (getOffsetX (startLine, startOffset), getOffsetY (startLine), width (), getOffsetY (startLine + 1));
            if (startLine < endLine - 1)
                paintRegion (0, getOffsetY (startLine + 1), width (), getOffsetY (endLine));
            paintRegion (0, getOffsetY (endLine), getOffsetX (endLine, endOffset) + 1, getOffsetY (endLine + 1));
        }
        else
            paintRegion (getOffsetX (startLine, startOffset), getOffsetY (startLine), getOffsetX (endLine, endOffset) + 1, getOffsetY (endLine + 1));
    }
    
    /** Dirty the view for the region that a caret at this indexed document line and character offset would cover. */
    void dirtyCaret (int line, int offset)
    {
        int x, y;
        
        findOffset (line, offset, x, y);
        paintRegion (x + 0 - document.hscroll, y, x + 3 - document.hscroll, y + global.fontHeight);
    }
    
    /** Dirty the view for a single indexed document line. */
    void dirtyChangeLine (int line)
    {
        paintRegion (0, getOffsetY (line), width (), getOffsetY (line + 1));
    }

    /** Dirty the view from this indexed document line forward. */    
    void dirtyFromLine (int line)
    {
        paintRegion (0, getOffsetY (document.line), width (), height ());
    }
    
    void doKeyDown (Event e)
    {
        charArray [char []] bind;

        if (e.shift)
        {
            if (e.control)
                bind = shiftControlBindings;
            else
                bind = shiftBindings;
        }
        else if (e.control)
            bind = controlBindings;
        else
            bind = bindings;

        if (e.keyCode in bind)
            run (bind [e.keyCode]);
    }

    bit visibleModified = false;

    void boundPoint (inout int line, inout int offset)
    {
        line = imid (0, line, document.lines.length - 1);
        offset = imid (0, offset, document.lines [line].length);
    }
    
    /** Assign the vertical scrollbar parameters. */
    void vscrollAssign ()
    {
        vscrollPoint (document.lineOffset);
        vscrollPage (height () / global.fontHeight);
        vscrollRange (0, imax (document.lines.length, height () / global.fontHeight - 1));
    }

    void doPaint (Event event)
    {
        alias document d;
        int spaceWidth;
        bit sv = document.selExists ();
        bit focus = isFocus ();
        bit showCursor = ((elapsedTime () - caretTime) % 1000) < 500 || !focus;
        TabParams tp = document.tabParams ();
        
        visibleLineOffset = document.lineOffset;
        visibleLine = document.line;
        visibleOffset = document.offset;
        visibleSelStartLine = document.selStartLine;
        visibleSelEndLine = document.selEndLine;
        visibleSelStartOffset = document.selStartOffset;
        visibleSelEndOffset = document.selEndOffset;

        d.hscroll = imax (0, imin (d.hscroll, width () * 9));
        hscrollPage (width ());
        hscrollRange (0, width () * 10);

        vscrollAssign ();

        if (d.lineOffset >= d.lines.length || d.lineOffset < 0)
        {
            d.lineOffset = imin (d.lineOffset, d.lines.length - 1);
            d.lineOffset = imax (0, d.lineOffset);
            vscrollPoint (d.lineOffset);
        }

        if (sv)
        {
            boundPoint (d.selStartLine, d.selStartOffset);
            boundPoint (d.selEndLine, d.selEndOffset);
        }

        boundPoint (d.line, d.offset);
        global.statusBar.paintCheck ();

        int sl = d.selStartLine, so = d.selStartOffset;
        int el = d.selEndLine, eo = d.selEndOffset;

        if (visibleModified != d.modified)
            setCaption ();

        beginPaint ();
        if (!visible ())
            goto done;

        clear (global.background);
        font (global.font);
        spaceWidth = textWidth (" ") * 1.5;

        pageHeight = height () / global.fontHeight;

        int h = height ();

        document.cleanRange (0, d.lineOffset);
        bit more;
        int y;

        for (y = 0; y < document.lines.length; y ++)
        {
            int index = d.lineOffset + y;

            if (y * global.fontHeight > h || y * global.fontHeight > event.bottom || index >= d.lines.length)
                break;
            more = false;
            if ((y + 1) * global.fontHeight <= event.top)
                continue;

            /* Show the start of the selection. */
            if (sv && index >= sl && index <= el)
            {
                Color color = global.colorSelection;
                char [] text = d.lines [index];
                int start, end;

                if (index == sl)
                    start = getOffsetX (sl, so);
                else
                    start = baseX;

                if (index == el)
                    end = getOffsetX (el, eo);
                else
                    end = getOffsetX (index, text.length);

                color = color.blend (global.background, focus ? 0 : 128);

                penClear ();
                brushColor (color);
                rect (start - d.hscroll, y * global.fontHeight, end + 2 - d.hscroll, (y + 1) * global.fontHeight + 1);
            }

            /* Show the cursor. */
            if (showCursor && index == d.line)
            {
                Color color = global.colorCursor;
                int x, y;

                if (!focus)
                    color = color.blend (global.background, 128);
    
                findOffset (d.line, d.offset, x, y);
                penClear ();
                brushColor (color);
                rect (x + 0 - d.hscroll, y, x + 3 - d.hscroll, y + global.fontHeight);
            }

            more = document.cleanLine (index);
            char [] text = document.lines [index];
            char [] high = document.highs [index];
            int c, s;
            int x = baseX;

            for ( ; c < text.length; c ++)
            {
                if (std.ctype.isspace (text [c]))
                {
                    textColor (highColor (high [s]));
                    textPrint (x - d.hscroll, y * global.fontHeight, text [s .. c]);
                    x += textWidth (text [s .. c]);
                    s = c + 1;

                    if (text [c] == ' ')
                        x += spaceWidth;
                    else if (text [c] == '\t')
                        x += spaceWidth * tp.tabSize;
                }
                else if (high [c] != high [s])
                {
                    textColor (highColor (high [s]));
                    textPrint (x - d.hscroll, y * global.fontHeight, text [s .. c]);
                    x += textWidth (text [s .. c]);
                    s = c;
                }
            }

            textColor (highColor (high [s]));
            textPrint (x - d.hscroll, y * global.fontHeight, text [s .. c]);
        }
        
    done:
        endPaint ();
        document.dirty = false;
        dirty = false;
        
        if (more)
        {
            int index = d.lineOffset + y - 1;
            int start = index;
            
            while (index < d.lines.length)
            {
                if (getOffsetY (index) >= height ())
                    break;
                index ++;
                if (!document.cleanLine (index - 1))
                    break;
            } 

            paintRegion (0, getOffsetY (start), width (), getOffsetY (index));
        }
    }

    void repaintCaret (Event e)
    {
        dirtyCaret (document.line, document.offset);
        timer (500, &repaintCaret);
    }

    /* Set the line offset. */
    void setLineOffset (int index)
    {
        if (index < 0)
            index = 0;
        if (index >= document.lines.length)
            index = document.lines.length - 1;
        if (document.lineOffset == index)
            return;
        document.lineOffset = index;
        vscrollPoint (document.lineOffset);
        dirty = true;
    }

    void setCaption ()
    {
        Control parent = this.parent ();

        while (parent.parent () !== null)
            parent = parent.parent ();

        char [] t;

        t = document.name ();
        if (document.modified)
            t ~= " *";
        t ~= " - D Editor";

        (cast (Frame) parent).caption (t);
        visibleModified = document.modified;
        global.projectView.paint ();
    }

    static void addCommand (char [] name, char [] options, Command command, char [] desc, char [] help)
    {
        commands [name] = command;
        commandDescs [name] = desc;
        commandOptions [name] = options;
        commandHelp [name] = help;
    }

    void bind (char [] name, char [] code, bit shift, bit control)
    {
        if (shift)
        {
            if (control)
                shiftControlBindings [code] = name;
            else
                shiftBindings [code] = name;
        }
        else if (control)
            controlBindings [code] = name;
        else
            bindings [code] = name;
    }

    void bind (char [] command, char [] code)
    {
        bind (command, code, false, false);
    }

    override void paint ()
    {
        if (paintDefer)
            paintCalled = true;
        else
            super.paint ();
    }

    void run (char [] command)
    {
        bit recursed = runRecurse;
        char [] options = commandOptions [command];
        
        typing = false;

        assert (command in commands);

        if (!recursed && std.string.find (options, 'u') < 0)
        {
            if (document !== null)
                document.mark (document.line, document.offset);
        }

        runRecurse = true;
        paintDefer = true;
        
        if (macroRecording && std.string.find (options, 'm') < 0)
            macroNext ~= command;

        commands [command] (this);

        runRecurse = recursed;
        if (!recursed)
        {
            paintDefer = false;
            if (paintCalled)
                paint ();
            paintCalled = false;
            document.undoClean ();
        }
    }
    
    void runChar (char ch)
    {
        char [] text, result;
        bit recursed = runRecurse;
        
        if (ch < 32)
            return;
        if (!recursed && !typing)
            document.mark (document.line, document.offset);

        runRecurse = true;
        paintDefer = true;

        run ("DeleteSelection");
        typing = true;

        text = document.lines [document.line];
        result = text [0 .. document.offset].dup;
        result ~= ch;
        result ~= text [document.offset .. text.length];
        document.offset ++;
        document.setLine (document.line, result);
        caretChange ();

        runRecurse = recursed;
        if (!recursed)
        {
            paintDefer = false;
            if (paintCalled)
                paint ();
            paintCalled = false;
            document.undoClean ();
        }
        
        if (macroRecording)
            macroNext ~= fmt ("*%c", ch);
    }

    Menu returnMenu; /* Commands which return menus put their result in here. */

    void loadFile (char [] filename)
    {
        for (int c; c < documents.length; c ++)
            if (documents [c].filename == filename)
            {
                switchDocument (documents [c]);
                return;
            }

        Document base = document;
        Document d;

        d = new Document (filename);
        documents ~= d;
        documents.sort;
        ordered ~= d;

        currentFolder ().add (new ProjectView.DocumentRow (d));
        switchDocument (d);
    }

    void switchDocument (Document doc)
    {
        document = doc;
        fCurrentFolder = null;
        if (!global.projectView.deferPaint)
            setCaption ();
        global.projectView.paintDocument (doc);
        vscrollAssign ();
        paint ();
    }

    void newDocument ()
    {
        Document next;

        next = new Document ();
        documents ~= next;
        documents.sort;
        ordered ~= next;

        currentFolder ().add (new ProjectView.DocumentRow (next));
        global.projectView.paint ();
        switchDocument (next);
    }

    void closeDocument (Document document)
    {
        int index;

        global.documentRemoved (document);
        
        for (int c; c < documents.length; c ++)
            if (documents [c] == document)
            {
                for (int d = c; d < documents.length - 1; d ++)
                    documents [d] = documents [d + 1];
                index = 0;
                documents = documents [0 .. documents.length - 1];
                break;
            }

        for (int c; c < ordered.length; c ++)
            if (ordered [c] == document)
            {
                for (int d = c; d < ordered.length - 1; d ++)
                    ordered [d] = ordered [d + 1];
                index = 0;
                ordered = ordered [0 .. ordered.length - 1];
                break;
            }

        if (ordered.length && document === this.document)
            switchDocument (ordered [0]);
        else if (!ordered.length)
            newDocument ();
    }

    class onRunner
    {
        View view;
        char [] name;

        this (View view, char [] name)
        {
            this.view = view;
            this.name = name;
        }

        void notify (Event e)
        {
            view.run (name);
        }
    }

    /* Create an event delegate that runs the command on notification. */
    Dispatcher.Method onRun (char [] name)
    {
        onRunner runner = new onRunner (this, name);

        return &runner.notify;
    }

    void doMakeFocus (Event e)
    {
        makeFocus ();
        document.line = findNearestY (e.y);
        document.offset = findNearestX (document.line, e.x);
        document.selClear ();
        caretChange ();
    }
    
    void doLButtonDown (Event e)
    {
        makeFocus ();
        document.line = findNearestY (e.y);
        document.offset = findNearestX (document.line, e.x);
        document.selClear ();
        caretChange ();
        captureMouse ();
    }

    void doLButtonUp (Event e)
    {
        if (!isCaptor ())
            return;
        releaseMouse ();
    }

    void doMouseMove (Event e)
    {
        if (!isCaptor ())
            return;

        if (inClientRegion (e.x, e.y))
        {
            int oline = document.line, ooffset = document.offset;

            document.line = findNearestY (e.y);
            document.offset = findNearestX (document.line, e.x);
            document.selFollowCursor (oline, ooffset, document.line, document.offset);
            caretChange ();
        }
    }

    void doContextMenu (Event e)
    {
        menu.run (this);
    }

    void doMouseWheel (Event e)
    {
        int offset = e.wheel * height () / global.fontHeight / 6;

        if (offset < 0 && offset > -1)
            offset = -1;
        else if (offset > 0 && offset < 1)
            offset = 1;
        setLineOffset (document.lineOffset - offset);
        clean ();
    }

    void doVScroll (Event e)
    {
        alias document d;
        int offset;

        switch (e.scrollType)
        {
            case e.Scroll.Top:
                setLineOffset (0);
                break;

            case e.Scroll.Bottom:
                setLineOffset (d.lines.length - height () / global.fontHeight);
                break;

            case e.Scroll.LineUp:
                setLineOffset (d.lineOffset - 1);
                break;

            case e.Scroll.LineDown:
                setLineOffset (d.lineOffset + 1);
                break;

            case e.Scroll.PageDown:
                setLineOffset (d.lineOffset + imax (1, height () / global.fontHeight - 4));
                break;

            case e.Scroll.PageUp:
                setLineOffset (d.lineOffset - imax (1, height () / global.fontHeight - 4));
                break;

            case e.Scroll.Drop:
            case e.Scroll.Track:
                setLineOffset (e.scrollPoint);
                break;

            case e.Scroll.End:
                break;
        }

        clean ();
    }

    void doHScroll (Event e)
    {
        alias document d;

        switch (e.scrollType)
        {
            case e.Scroll.Top: d.hscroll = 0; break;
            case e.Scroll.Bottom: d.hscroll = width () * 9; break;
            case e.Scroll.LineUp: d.hscroll -= 16; break;
            case e.Scroll.LineDown: d.hscroll += 16; break;
            case e.Scroll.PageDown: d.hscroll += width () * 0.9; break;
            case e.Scroll.PageUp: d.hscroll -= width () * 0.9; break;
            case e.Scroll.Drop:
            case e.Scroll.Track:
                d.hscroll = e.scrollPoint;
                break;
            case e.Scroll.End:
                break;
        }

        paint ();
    }
}
